<?php
include('includes/header.php'); 
$error = '';
$success = '';

// 1. Get User ID from URL
if (!isset($_GET['id']) || empty($_GET['id'])) {
    // If no ID is provided, redirect back to user list
    header('Location: users.php');
    exit;
}

$user_id = (int)$_GET['id'];
$pdo = get_pdo();
$user = null;

if ($pdo) {
    try {
        // Fetch current user data
        $stmt = $pdo->prepare("SELECT id, username, email, role FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$user) {
            // User not found
            header('Location: users.php?status=danger&msg=' . urlencode('User not found.'));
            exit;
        }

    } catch (PDOException $e) {
        $error = 'Database error fetching user details.';
    }
} else {
    $error = 'Database connection failed.';
}

// 2. Handle Form Submission (Update Logic)
if ($_SERVER['REQUEST_METHOD'] == 'POST' && $user) {
    // Get updated data from form
    $new_username = trim($_POST['username']);
    $new_email = trim($_POST['email']);
    $new_role = $_POST['role'];
    $new_password = $_POST['password']; // Optional password change

    // --- Validation ---
    if (empty($new_username) || empty($new_email)) {
        $error = 'Username and Email fields are required.';
    } elseif (!filter_var($new_email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Invalid email format.';
    } else {
        // --- Prepare SQL Update ---
        $update_fields = ['username' => $new_username, 'email' => $new_email, 'role' => $new_role];
        $sql_parts = [];
        $params = [];
        
        // Check for duplicate username/email (excluding current user)
        $check_stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE (username = ? OR email = ?) AND id != ?");
        $check_stmt->execute([$new_username, $new_email, $user_id]);

        if ($check_stmt->fetchColumn() > 0) {
            $error = 'Username or Email already taken by another user.';
        } else {
            // Build SQL for mandatory fields
            $sql_parts[] = 'username = ?';
            $params[] = $new_username;
            $sql_parts[] = 'email = ?';
            $params[] = $new_email;
            $sql_parts[] = 'role = ?';
            $params[] = $new_role;

            // Handle optional password change
            if (!empty($new_password)) {
                if (strlen($new_password) < 6) {
                    $error = 'New password must be at least 6 characters long.';
                } else {
                    $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                    $sql_parts[] = 'password = ?';
                    $params[] = $hashed_password;
                }
            }

            // If no validation error occurred on password
            if (empty($error)) {
                try {
                    $sql = "UPDATE users SET " . implode(', ', $sql_parts) . " WHERE id = ?";
                    $params[] = $user_id;

                    $update_stmt = $pdo->prepare($sql);
                    $update_stmt->execute($params);

                    $success = "User **" . htmlspecialchars($new_username) . "** updated successfully!";
                    
                    // Re-fetch updated user data for form display
                    $stmt = $pdo->prepare("SELECT id, username, email, role FROM users WHERE id = ?");
                    $stmt->execute([$user_id]);
                    $user = $stmt->fetch(PDO::FETCH_ASSOC);

                } catch (PDOException $e) {
                    $error = 'Database error updating user: ' . $e->getMessage();
                }
            }
        }
    }
}

// Fallback data for form (use POST data if available after a failed submission)
$form_data = $_POST ? $_POST : $user;
?>

<h1 class="mb-4"><i class="fas fa-edit me-2"></i> Edit User: <?= htmlspecialchars($user['username'] ?? 'N/A') ?></h1>
<hr>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $error ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<div class="card shadow mb-4">
    <div class="card-header py-3">
        <h6 class="m-0 font-weight-bold text-primary">Edit Details for ID: <?= $user_id ?></h6>
    </div>
    <div class="card-body">
        <form method="POST">
            <div class="row">
                <div class="col-md-6 mb-3">
                    <label for="username" class="form-label">Username <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" id="username" name="username" 
                           value="<?= htmlspecialchars($form_data['username'] ?? '') ?>" required>
                </div>
                <div class="col-md-6 mb-3">
                    <label for="email" class="form-label">Email <span class="text-danger">*</span></label>
                    <input type="email" class="form-control" id="email" name="email" 
                           value="<?= htmlspecialchars($form_data['email'] ?? '') ?>" required>
                </div>
            </div>
            
            <div class="row">
                <div class="col-md-6 mb-3">
                    <label for="password" class="form-label">New Password (Optional)</label>
                    <input type="password" class="form-control" id="password" name="password">
                    <small class="text-muted">Leave blank to keep the current password.</small>
                </div>
                <div class="col-md-6 mb-3">
                    <label for="role" class="form-label">User Role</label>
                    <select class="form-select" id="role" name="role">
                        <?php $current_role = $form_data['role'] ?? 'user'; ?>
                        <option value="user" <?= ($current_role == 'user') ? 'selected' : '' ?>>User</option>
                        <option value="editor" <?= ($current_role == 'editor') ? 'selected' : '' ?>>Editor</option>
                        <option value="admin" <?= ($current_role == 'admin') ? 'selected' : '' ?>>Admin</option>
                    </select>
                </div>
            </div>

            <div class="d-flex justify-content-between">
                <a href="users.php" class="btn btn-secondary">
                    <i class="fas fa-arrow-left me-1"></i> Back to User List
                </a>
                <button type="submit" class="btn btn-success">
                    <i class="fas fa-sync-alt me-1"></i> Update User
                </button>
            </div>
        </form>
    </div>
</div>

<?php include('includes/footer.php'); ?>