<?php
include('includes/header.php'); // Includes session check and security
$error = '';
$success = '';

// Check if the form is submitted
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    $role = $_POST['role'] ?? 'user'; // Default role is 'user'

    // --- Validation ---
    if (empty($username) || empty($email) || empty($password)) {
        $error = 'All required fields must be filled.';
    } elseif ($password !== $confirm_password) {
        $error = 'Passwords do not match.';
    } elseif (strlen($password) < 6) {
        $error = 'Password must be at least 6 characters long.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Invalid email format.';
    } else {
        // --- Database Insertion ---
        $pdo = get_pdo();

        if ($pdo) {
            try {
                // 1. Check if username or email already exists
                $check_stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE username = ? OR email = ?");
                $check_stmt->execute([$username, $email]);
                
                if ($check_stmt->fetchColumn() > 0) {
                    $error = 'Username or Email already exists. Please choose a different one.';
                } else {
                    // 2. Hash the password
                    $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                    
                    // 3. Insert the new user
                    $stmt = $pdo->prepare("INSERT INTO users (username, email, password, role) VALUES (?, ?, ?, ?)");
                    $stmt->execute([$username, $email, $hashed_password, $role]);
                    
                    $success = "User **" . htmlspecialchars($username) . "** added successfully!";
                    
                    // Optional: Clear POST data after success
                    $_POST = array(); 
                }

            } catch (PDOException $e) {
                // Log and display error
                $error = 'Database error: Could not add user. ' . $e->getMessage();
            }
        } else {
            $error = 'Database connection failed.';
        }
    }
}
?>

<h1 class="mb-4"><i class="fas fa-user-plus me-2"></i> Add New User</h1>
<hr>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $error ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<div class="card shadow mb-4">
    <div class="card-header py-3">
        <h6 class="m-0 font-weight-bold text-primary">User Details Form</h6>
    </div>
    <div class="card-body">
        <form method="POST">
            <div class="row">
                <div class="col-md-6 mb-3">
                    <label for="username" class="form-label">Username <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" id="username" name="username" 
                           value="<?= htmlspecialchars($_POST['username'] ?? '') ?>" required>
                </div>
                <div class="col-md-6 mb-3">
                    <label for="email" class="form-label">Email <span class="text-danger">*</span></label>
                    <input type="email" class="form-control" id="email" name="email" 
                           value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" required>
                </div>
            </div>
            
            <div class="row">
                <div class="col-md-6 mb-3">
                    <label for="password" class="form-label">Password <span class="text-danger">*</span></label>
                    <input type="password" class="form-control" id="password" name="password" required>
                </div>
                <div class="col-md-6 mb-3">
                    <label for="confirm_password" class="form-label">Confirm Password <span class="text-danger">*</span></label>
                    <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                </div>
            </div>

            <div class="mb-3">
                <label for="role" class="form-label">User Role</label>
                <select class="form-select" id="role" name="role">
                    <option value="user" <?= (($_POST['role'] ?? '') == 'user') ? 'selected' : '' ?>>User</option>
                    <option value="editor" <?= (($_POST['role'] ?? '') == 'editor') ? 'selected' : '' ?>>Editor</option>
                    <option value="admin" <?= (($_POST['role'] ?? '') == 'admin') ? 'selected' : '' ?>>Admin</option>
                </select>
            </div>

            <div class="d-flex justify-content-between">
                <a href="users.php" class="btn btn-secondary">
                    <i class="fas fa-arrow-left me-1"></i> Back to User List
                </a>
                <button type="submit" class="btn btn-success">
                    <i class="fas fa-save me-1"></i> Create User
                </button>
            </div>
        </form>
    </div>
</div>

<?php include('includes/footer.php'); ?>