<?php
include('includes/header.php'); 

$error = '';
$success = '';

// Get the ID of the currently logged-in user
$user_id = $_SESSION['user_id']; 
$pdo = get_pdo();
$user = null;

if ($pdo) {
    try {
        // Fetch current user data (exclude sensitive password hash from the fetch)
        $stmt = $pdo->prepare("SELECT id, username, email, role FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$user) {
            // Should not happen if login check is working, but safety first
            header('Location: logout.php');
            exit;
        }

    } catch (PDOException $e) {
        $error = 'Database error fetching profile details.';
    }
} else {
    $error = 'Database connection failed.';
}


// 2. Handle Form Submission (Update Logic)
if ($_SERVER['REQUEST_METHOD'] == 'POST' && $user) {
    // Get updated data from form
    $new_username = trim($_POST['username']);
    $new_email = trim($_POST['email']);
    $new_password = $_POST['password']; // Optional password change

    // --- Validation ---
    if (empty($new_username) || empty($new_email)) {
        $error = 'Username and Email fields are required.';
    } elseif (!filter_var($new_email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Invalid email format.';
    } else {
        // --- Prepare SQL Update ---
        $sql_parts = [];
        $params = [];
        
        // Check for duplicate username/email (excluding current user)
        $check_stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE (username = ? OR email = ?) AND id != ?");
        $check_stmt->execute([$new_username, $new_email, $user_id]);

        if ($check_stmt->fetchColumn() > 0) {
            $error = 'Username or Email already taken by another user.';
        } else {
            // Build SQL for mandatory fields
            $sql_parts[] = 'username = ?';
            $params[] = $new_username;
            $sql_parts[] = 'email = ?';
            $params[] = $new_email;

            // Handle optional password change
            if (!empty($new_password)) {
                if (strlen($new_password) < 6) {
                    $error = 'New password must be at least 6 characters long.';
                } else {
                    $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                    $sql_parts[] = 'password = ?';
                    $params[] = $hashed_password;
                }
            }

            // If no validation error occurred on password
            if (empty($error)) {
                try {
                    $sql = "UPDATE users SET " . implode(', ', $sql_parts) . " WHERE id = ?";
                    $params[] = $user_id;

                    $update_stmt = $pdo->prepare($sql);
                    $update_stmt->execute($params);

                    $success = "Profile updated successfully!";
                    
                    // Re-fetch updated user data and update session username if it changed
                    $_SESSION['username'] = $new_username; 
                    $stmt = $pdo->prepare("SELECT id, username, email, role FROM users WHERE id = ?");
                    $stmt->execute([$user_id]);
                    $user = $stmt->fetch(PDO::FETCH_ASSOC);

                } catch (PDOException $e) {
                    $error = 'Database error updating profile: ' . $e->getMessage();
                }
            }
        }
    }
}

// Fallback data for form (use POST data if available after a failed submission)
$form_data = $_POST ? array_merge($user, $_POST) : $user;
?>

<h1 class="mb-4"><i class="fas fa-user-circle me-2"></i> Edit Your Profile</h1>
<hr>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $error ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<div class="card shadow mb-4">
    <div class="card-header py-3">
        <h6 class="m-0 font-weight-bold text-primary">Details for: <?= htmlspecialchars($user['username'] ?? 'N/A') ?> (Role: <?= ucfirst(htmlspecialchars($user['role'] ?? 'N/A')) ?>)</h6>
    </div>
    <div class="card-body">
        <form method="POST">
            <div class="mb-3">
                <label for="username" class="form-label">Username <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="username" name="username" 
                       value="<?= htmlspecialchars($form_data['username'] ?? '') ?>" required>
            </div>
            <div class="mb-3">
                <label for="email" class="form-label">Email <span class="text-danger">*</span></label>
                <input type="email" class="form-control" id="email" name="email" 
                       value="<?= htmlspecialchars($form_data['email'] ?? '') ?>" required>
            </div>
            
            <div class="mb-3">
                <label for="password" class="form-label">New Password (Optional)</label>
                <input type="password" class="form-control" id="password" name="password">
                <small class="text-muted">Leave blank to keep your current password.</small>
            </div>
            
            <div class="mb-3">
                <label for="role" class="form-label">Your Current Role</label>
                <input type="text" class="form-control" value="<?= ucfirst(htmlspecialchars($user['role'] ?? 'N/A')) ?>" disabled>
                <small class="text-muted">Contact an Admin to change your role.</small>
            </div>

            <button type="submit" class="btn btn-success">
                <i class="fas fa-save me-1"></i> Save Profile Changes
            </button>
        </form>
    </div>
</div>

<?php include('includes/footer.php'); ?>