<?php
include('includes/header.php'); 

$error = '';
$success = '';

// 1. Get Page ID from URL
if (!isset($_GET['id']) || empty($_GET['id'])) {
    header('Location: pages.php');
    exit;
}

$page_id = (int)$_GET['id'];
$pdo = get_pdo();
$page = null;

if ($pdo) {
    try {
        // Fetch current page data
        $stmt = $pdo->prepare("SELECT id, title, slug, content, status FROM pages WHERE id = ?");
        $stmt->execute([$page_id]);
        $page = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$page) {
            header('Location: pages.php?status=danger&msg=' . urlencode('Page not found.'));
            exit;
        }

    } catch (PDOException $e) {
        $error = 'Database error fetching page details.';
    }
} else {
    $error = 'Database connection failed.';
}


// 2. Handle Form Submission (Update Logic)
if ($_SERVER['REQUEST_METHOD'] == 'POST' && $page) {
    // Get updated data from form
    $new_title = trim($_POST['title']);
    $new_slug = trim($_POST['slug']);
    $new_content = $_POST['content'];
    $new_status = $_POST['status'];

    // --- Validation ---
    if (empty($new_title) || empty($new_slug)) {
        $error = 'Title and Slug fields are required.';
    } else {
        // --- Database Update ---
        try {
            // Check for duplicate slug (excluding current page)
            $check_stmt = $pdo->prepare("SELECT COUNT(*) FROM pages WHERE slug = ? AND id != ?");
            $check_stmt->execute([$new_slug, $page_id]);

            if ($check_stmt->fetchColumn() > 0) {
                $error = 'The Slug already exists. Please choose a different one.';
            } else {
                // Update the page
                $sql = "UPDATE pages SET title = ?, slug = ?, content = ?, status = ? WHERE id = ?";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([$new_title, $new_slug, $new_content, $new_status, $page_id]);

                $success = "Page **" . htmlspecialchars($new_title) . "** updated successfully!";
                
                // Re-fetch updated data
                $stmt = $pdo->prepare("SELECT id, title, slug, content, status FROM pages WHERE id = ?");
                $stmt->execute([$page_id]);
                $page = $stmt->fetch(PDO::FETCH_ASSOC);
            }

        } catch (PDOException $e) {
            $error = 'Database error updating page: ' . $e->getMessage();
        }
    }
}

// Fallback data for form (use POST data if available after a failed submission, else use fetched $page)
$form_data = $_POST ? array_merge($page, $_POST) : $page;
?>

<h1 class="mb-4"><i class="fas fa-edit me-2"></i> Edit Page: <?= htmlspecialchars($page['title'] ?? 'N/A') ?></h1>
<hr>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $error ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<div class="card shadow mb-4">
    <div class="card-header py-3">
        <h6 class="m-0 font-weight-bold text-primary">Edit Details for ID: <?= $page_id ?></h6>
    </div>
    <div class="card-body">
        <form method="POST">
            <div class="mb-3">
                <label for="title" class="form-label">Page Title <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="title" name="title" 
                       value="<?= htmlspecialchars($form_data['title'] ?? '') ?>" required>
            </div>
            
            <div class="mb-3">
                <label for="slug" class="form-label">Page Slug (URL) <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="slug" name="slug" 
                       value="<?= htmlspecialchars($form_data['slug'] ?? '') ?>" required>
                <small class="text-muted">Example: about-us. Only lowercase letters, numbers, and hyphens.</small>
            </div>

            <div class="mb-3">
                <label for="content" class="form-label">Content</label>
                <textarea class="form-control" id="content" name="content" rows="10"><?= htmlspecialchars($form_data['content'] ?? '') ?></textarea>
            </div>

            <div class="mb-3">
                <label for="status" class="form-label">Status</label>
                <select class="form-select" id="status" name="status">
                    <option value="draft" <?= (($form_data['status'] ?? '') == 'draft') ? 'selected' : '' ?>>Draft</option>
                    <option value="published" <?= (($form_data['status'] ?? '') == 'published') ? 'selected' : '' ?>>Published</option>
                </select>
            </div>

            <div class="d-flex justify-content-between">
                <a href="pages.php" class="btn btn-secondary">
                    <i class="fas fa-arrow-left me-1"></i> Back to Page List
                </a>
                <button type="submit" class="btn btn-success">
                    <i class="fas fa-sync-alt me-1"></i> Update Page
                </button>
            </div>
        </form>
    </div>
</div>

<?php include('includes/footer.php'); ?>