<?php
include('includes/header.php'); 

$error = '';
$success = '';

// 1. Get Item ID from URL
if (!isset($_GET['item_id']) || empty($_GET['item_id'])) {
    // If no item_id, redirect back to the main menus list or items list
    header('Location: menus.php');
    exit;
}

$item_id = (int)$_GET['item_id'];
$pdo = get_pdo();
$item = null;
$menu_items_for_parent = []; // For the parent dropdown
$current_menu_id = 0; // To keep track of which menu we are in

if ($pdo) {
    try {
        // Fetch current item data
        $stmt = $pdo->prepare("SELECT id, menu_id, title, url, parent_id FROM menu_items WHERE id = ?");
        $stmt->execute([$item_id]);
        $item = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$item) {
            header('Location: menus.php?status=danger&msg=' . urlencode('Menu Item not found.'));
            exit;
        }

        $current_menu_id = $item['menu_id'];

        // Fetch all top-level items from the same menu for the Parent dropdown
        $stmt_parents = $pdo->prepare("SELECT id, title, parent_id FROM menu_items WHERE menu_id = ? AND id != ? AND parent_id IS NULL ORDER BY order_num ASC");
        $stmt_parents->execute([$current_menu_id, $item_id]); // Exclude current item itself
        $menu_items_for_parent = $stmt_parents->fetchAll(PDO::FETCH_ASSOC);

    } catch (PDOException $e) {
        $error = 'Database error fetching menu item details: ' . $e->getMessage();
    }
} else {
    $error = 'Database connection failed.';
}


// 2. Handle Form Submission (Update Logic)
if ($_SERVER['REQUEST_METHOD'] == 'POST' && $item) {
    // Get updated data from form
    $new_title = trim($_POST['item_title']);
    $new_url = trim($_POST['item_url']);
    // If parent_id is empty, set it to NULL
    $new_parent_id = empty($_POST['parent_id']) ? NULL : (int)$_POST['parent_id']; 

    // --- Validation ---
    if (empty($new_title) || empty($new_url)) {
        $error = 'Title and URL fields are required.';
    } else {
        // --- Database Update ---
        try {
            $sql = "UPDATE menu_items SET title = ?, url = ?, parent_id = ? WHERE id = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$new_title, $new_url, $new_parent_id, $item_id]);

            $success = "Menu Item **" . htmlspecialchars($new_title) . "** updated successfully!";
            
            // Re-fetch updated data to display in the form
            $stmt = $pdo->prepare("SELECT id, menu_id, title, url, parent_id FROM menu_items WHERE id = ?");
            $stmt->execute([$item_id]);
            $item = $stmt->fetch(PDO::FETCH_ASSOC);

        } catch (PDOException $e) {
            $error = 'Database error updating menu item: ' . $e->getMessage();
        }
    }
}

// Fallback data for form (use POST data if available after a failed submission, else use fetched $item)
$form_data = $_POST ? array_merge($item, $_POST) : $item;
?>

<h1 class="mb-4"><i class="fas fa-edit me-2"></i> Edit Menu Item: **<?= htmlspecialchars($item['title'] ?? 'N/A') ?>**</h1>
<hr>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $error ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<div class="card shadow mb-4">
    <div class="card-header py-3">
        <h6 class="m-0 font-weight-bold text-primary">Edit Item Details (ID: <?= $item_id ?>)</h6>
    </div>
    <div class="card-body">
        <form method="POST">
            <div class="mb-3">
                <label for="item_title" class="form-label">Link Text <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="item_title" name="item_title" 
                       value="<?= htmlspecialchars($form_data['title'] ?? '') ?>" required>
            </div>
            
            <div class="mb-3">
                <label for="item_url" class="form-label">URL / Link <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="item_url" name="item_url" 
                       value="<?= htmlspecialchars($form_data['url'] ?? '') ?>" required>
            </div>

            <div class="mb-3">
                <label for="parent_id" class="form-label">Parent Item (For Sub-menu)</label>
                <select class="form-select" id="parent_id" name="parent_id">
                    <option value="" <?= (empty($form_data['parent_id'])) ? 'selected' : '' ?>>-- None (Top Level) --</option>
                    <?php 
                    $current_parent_id = $form_data['parent_id'] ?? null;
                    foreach ($menu_items_for_parent as $parent_item): 
                    ?>
                        <option value="<?= $parent_item['id'] ?>" 
                            <?= ($current_parent_id == $parent_item['id']) ? 'selected' : '' ?>>
                            <?= htmlspecialchars($parent_item['title']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <small class="text-muted">Only top-level items from this menu are shown as potential parents.</small>
            </div>

            <div class="d-flex justify-content-between">
                <a href="menu_items.php?menu_id=<?= $current_menu_id ?>" class="btn btn-secondary">
                    <i class="fas fa-arrow-left me-1"></i> Back to Menu Items
                </a>
                <button type="submit" class="btn btn-success">
                    <i class="fas fa-sync-alt me-1"></i> Update Item
                </button>
            </div>
        </form>
    </div>
</div>

<?php include('includes/footer.php'); ?>