<?php
include('includes/header.php'); 

$error = '';
$success = '';
$pdo = get_pdo();
$files = [];

// Define the directory where files are stored (relative to admin folder)
$upload_dir = '../uploads/'; 
$max_file_size = 5 * 1024 * 1024; // 5MB limit
$allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'application/pdf']; 

// --- 1. Handle File Upload ---
if (isset($_POST['upload_file'])) {
    if (isset($_FILES['media_file']) && $_FILES['media_file']['error'] == UPLOAD_ERR_OK) {
        
        $file = $_FILES['media_file'];
        $filename = basename($file['name']);
        $mime_type = $file['type'];
        $file_size = $file['size'];
        $temp_path = $file['tmp_name'];
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        // Validation
        if ($file_size > $max_file_size) {
            $error = 'File size exceeds the 5MB limit.';
        } elseif (!in_array($mime_type, $allowed_types)) {
            $error = 'Invalid file type. Only JPG, PNG, GIF, and PDF are allowed.';
        } else {
            // Generate a unique filename to prevent overwriting
            $unique_filename = uniqid('file_', true) . '.' . $extension;
            $destination = $upload_dir . $unique_filename;
            
            if (move_uploaded_file($temp_path, $destination)) {
                
                // Save metadata to database
                try {
                    $stmt = $pdo->prepare("INSERT INTO files (filename, file_path, mime_type, file_size, uploaded_by) 
                                         VALUES (?, ?, ?, ?, ?)");
                    $stmt->execute([
                        $filename, 
                        $unique_filename, // Save unique path relative to uploads/
                        $mime_type, 
                        $file_size, 
                        $_SESSION['user_id']
                    ]);
                    $success = "File **" . htmlspecialchars($filename) . "** uploaded successfully!";
                } catch (PDOException $e) {
                    // If DB fails, try to delete the physical file
                    unlink($destination); 
                    $error = 'Error saving file metadata: ' . $e->getMessage();
                }
            } else {
                $error = 'Error moving file to the uploads directory.';
            }
        }
    } else {
        $error = 'No file selected or an upload error occurred.';
    }
}

// --- 2. Handle File Deletion ---
if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
    $file_id = (int)$_GET['id'];
    try {
        // Fetch file path before deleting record
        $stmt = $pdo->prepare("SELECT file_path FROM files WHERE id = ?");
        $stmt->execute([$file_id]);
        $file_data = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($file_data) {
            $file_to_delete = $upload_dir . $file_data['file_path'];
            
            // Delete record from DB
            $pdo->prepare("DELETE FROM files WHERE id = ?")->execute([$file_id]);

            // Delete physical file
            if (file_exists($file_to_delete)) {
                unlink($file_to_delete);
            }
            
            header('Location: media.php?status=success&msg=' . urlencode('File deleted successfully.'));
            exit;
        } else {
            $error = 'File not found in database.';
        }
    } catch (PDOException $e) {
        $error = 'Database error during file deletion: ' . $e->getMessage();
    }
}

// Display status message if redirected after action
if (isset($_GET['status']) && isset($_GET['msg'])) {
    $success = htmlspecialchars($_GET['msg']);
}


// --- 3. Fetch All Files for Display ---
if ($pdo) {
    try {
        $stmt = $pdo->prepare("SELECT f.id, f.filename, f.file_path, f.mime_type, f.file_size, f.uploaded_at, u.username 
                               FROM files f LEFT JOIN users u ON f.uploaded_by = u.id ORDER BY f.uploaded_at DESC");
        $stmt->execute();
        $files = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        $error = 'Error fetching files: ' . $e->getMessage();
    }
}
?>

<h1 class="mb-4"><i class="fas fa-photo-video me-2"></i> Media Library</h1>
<hr>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>
<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $error ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<div class="card shadow mb-4">
    <div class="card-header py-3 d-flex justify-content-between align-items-center">
        <h6 class="m-0 font-weight-bold text-primary">Upload New File</h6>
    </div>
    <div class="card-body">
        <form method="POST" enctype="multipart/form-data">
            <input type="hidden" name="upload_file" value="1">
            <div class="row">
                <div class="col-md-8">
                    <input type="file" class="form-control" name="media_file" required>
                    <small class="text-muted">Allowed: JPG, PNG, GIF, PDF. Max size: 5MB.</small>
                </div>
                <div class="col-md-4">
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="fas fa-upload me-1"></i> Upload
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>

<div class="card shadow mb-4">
    <div class="card-header py-3">
        <h6 class="m-0 font-weight-bold text-primary">All Uploaded Files (<?= count($files) ?>)</h6>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <?php if ($files): ?>
                <table class="table table-bordered table-striped" width="100%" cellspacing="0">
                    <thead>
                        <tr>
                            <th>Preview</th>
                            <th>Original Name</th>
                            <th>Type/Size</th>
                            <th>Uploaded By</th>
                            <th>Date</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($files as $file): ?>
                        <tr>
                            <td>
                                <?php 
                                $full_path = $upload_dir . htmlspecialchars($file['file_path']);
                                if (strpos($file['mime_type'], 'image') !== false): ?>
                                    <img src="<?= $full_path ?>" alt="<?= htmlspecialchars($file['filename']) ?>" style="max-height: 50px;">
                                <?php else: ?>
                                    <i class="fas fa-file me-2"></i> File
                                <?php endif; ?>
                            </td>
                            <td>
                                <strong><?= htmlspecialchars($file['filename']) ?></strong>
                                <br>
                                <small>URL: <code><?= $full_path ?></code></small>
                            </td>
                            <td>
                                <?= htmlspecialchars($file['mime_type']) ?>
                                <br>
                                <small><?= round($file['file_size'] / 1024, 2) ?> KB</small>
                            </td>
                            <td><?= htmlspecialchars($file['username'] ?? 'N/A') ?></td>
                            <td><?= date('Y-m-d H:i', strtotime($file['uploaded_at'])) ?></td>
                            <td>
                                <a href="media.php?action=delete&id=<?= $file['id'] ?>" 
                                   class="btn btn-sm btn-danger" 
                                   title="Delete File"
                                   onclick="return confirm('Are you sure you want to delete the file: <?= htmlspecialchars($file['filename']) ?>?');">
                                    <i class="fas fa-trash"></i>
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <div class="alert alert-info">No files have been uploaded yet.</div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include('includes/footer.php'); ?>